//package globalExamples;
import oracle.olapi.data.source.DataProvider;
import oracle.olapi.data.source.Source;
import oracle.olapi.data.source.Model;
import oracle.olapi.data.source.LiteralQualification;
import oracle.olapi.data.source.Qualification;
import oracle.olapi.data.source.NumberSource;
import oracle.olapi.data.source.StringSource;

import oracle.olapi.metadata.mdm.MdmAttribute;
import oracle.olapi.metadata.mdm.MdmLevel;
import oracle.olapi.metadata.mdm.MdmLevelHierarchy;
import oracle.olapi.metadata.mdm.MdmMeasure;
import oracle.olapi.metadata.mdm.MdmMeasureModel;
import oracle.olapi.metadata.mdm.MdmModel;
import oracle.olapi.metadata.mdm.MdmPrimaryDimension;
import oracle.olapi.metadata.mdm.MdmStandardDimension;
import oracle.olapi.metadata.mdm.MdmStandardMember;

import oracle.olapi.metadata.mdm.MdmDimensionCalculationModel;
import oracle.olapi.data.source.Assignment;
import oracle.olapi.data.source.LiteralQualification;
import oracle.olapi.data.source.Model;

/**
 * This program contains the code that appears in Example 6-13, Creating a
 * Custom Member That Assigns an Aggregated Value, in Chapter 6,
 * Understanding Source Objects, in the Oracle OLAP Developer's Guide to the
 * OLAP API.
 * This program creates a custom member of the Product dimension.
 * For the custom member Oracle OLAP automatically adds an Assignment object
 * to the Number MdmDimensionCalculationModel for the dimension.
 * The value that Oracle OLAP assigns for the custom member is the result of
 * an aggregation of the values specified by other dimension members.
 *
 * This program uses the Context10g class, which uses the
 * CursorPrintWriter class.
 * It also uses the CreateColorAttribute class, which generates a custom
 * MdmAttribute that relates colors to Product dimension members.
 *
 * @author Oracle Corporation
 */
public class CreateCustomMemberWithAggVal
{

  public void run(String[] args)
  {
    // Create a PrintWriter and objects for connecting to the database and
    // getting the metadata.
    Context10g context = new Context10g(args, false);

    context.println("Example 6-13, Creating a Custom Member That Assigns an "
                    + "Aggregated Value,\n");

    // Get the DataProvider.
    DataProvider dp = context.getDataProvider();

    // Create a color attribute for the Product dimension.
    CreateColorAttribute ccattr = new CreateColorAttribute();
    MdmAttribute mdmProdColorAttr = ccattr.getAttribute(context, dp);

    // Get the Source for the custom attribute.
    Source prodColorAttr = mdmProdColorAttr.getSource();

    // Get the Units measure and the Source for it.
    MdmMeasure mdmUnits = context.getMdmMeasureByName("UNITS");
    NumberSource units = (NumberSource) mdmUnits.getSource();

    // Get the dimensions. Cast the Product dimension as an MdmStandardDimension
    // so that it has a createCustomMember method.
    MdmStandardDimension mdmProdStdDim = (MdmStandardDimension)
                                     context.getMdmPrimaryDimensionByName("PRODUCT");
    MdmPrimaryDimension mdmTimeDim = context.getMdmPrimaryDimensionByName("TIME");
    MdmPrimaryDimension mdmChanDim = context.getMdmPrimaryDimensionByName("CHANNEL");
    MdmPrimaryDimension mdmCustDim = context.getMdmPrimaryDimensionByName("CUSTOMER");

    // Get the Source for the product dimension.
    Source prodStdDim = mdmProdStdDim.getSource();

    // Get the default hierarchies of the dimensions and the Source objects for them.
    MdmLevelHierarchy mdmCalendar = (MdmLevelHierarchy)
                                     mdmTimeDim.getDefaultHierarchy();
    MdmLevelHierarchy mdmProdRollup = (MdmLevelHierarchy)
                                       mdmProdStdDim.getDefaultHierarchy();
    MdmLevelHierarchy mdmChanRollup = (MdmLevelHierarchy)
                                       mdmChanDim.getDefaultHierarchy();
    MdmLevelHierarchy mdmShipRollup= (MdmLevelHierarchy)
                                      mdmCustDim.getDefaultHierarchy();

    StringSource calendar = (StringSource) mdmCalendar.getSource();
    StringSource prodRollup = (StringSource) mdmProdRollup.getSource();
    StringSource chanRollup = (StringSource) mdmChanRollup.getSource();
    StringSource shipRollup = (StringSource) mdmShipRollup.getSource();

    // Get the placeholder Source for the Number data type.
    Source ph = dp.getFundamentalMetadataProvider()
                  .getNumberPlaceholder()
                  .getSource();

    // Get the level to which the dimension members belong.
    MdmLevel mdmItemLevel = context.getLevelByName(mdmProdRollup, "ITEM");

    // Get the Source for the ITEM level.
    Source itemLevel = mdmItemLevel.getSource();

    // Create a Source that has the green products.
    Source greenProducts = itemLevel.join(prodColorAttr, "Green");

    // Get the short value description attribute for Product and the Source for it.
    MdmAttribute mdmProdShortDescr = mdmProdStdDim.getShortValueDescriptionAttribute();
    Source prodShortDescr = mdmProdShortDescr.getSource();

    // Produce a Source that has a short description for each product.
    Source prodByShortDescr = prodShortDescr.join(prodRollup);

    // Create the calculation Source, which specifies the aggregation of
    // the selected products.
    Source calc = ((NumberSource)
                   (ph.join(prodRollup,
                             new String[] {"PRODUCT_ROLLUP::ITEM::25",
                                           "PRODUCT_ROLLUP::ITEM::26",
                                           "PRODUCT_ROLLUP::ITEM::30",
                                           "PRODUCT_ROLLUP::ITEM::31"}))).total();

    // Create the custom dimension member.
    MdmStandardMember mdmGreenProdTotal =
           mdmProdStdDim.createCustomMember("65",  // member local value
                                            mdmItemLevel, // member level
                                            "4",   // parent local value
                                            calc, // calculation Source
                                            10);   // precedence value

    // Set the short value description for the custom member.
    mdmGreenProdTotal.setShortDescription("Green Products Total");

    // Append the custom member to the green products selection.
    Source greenProdWithTotal =
       greenProducts.appendValue(prodRollup.selectValue("PRODUCT_ROLLUP::ITEM::65"));

    // Create a query that specifies the units sold of the selected products,
    // including a short description of the products, for the specified customer
    // through all sales channels in the specified year.
    Source result = units.join(prodShortDescr.join(greenProdWithTotal))
                           .join(shipRollup, "SHIPMENTS_ROLLUP::SHIP_TO::106")
                           .join(chanRollup, "CHANNEL_ROLLUP::ALL_CHANNELS::1")
                           .join(calendar, "CALENDAR::YEAR::3");

    // Prepare and commit the current Transaction.
    context.commit();

    // Displays the values of a Cursor for result.
    context.displayResult(result);
  }

  public static void main(String[] args)
  {
    new CreateCustomMemberWithAggVal().run(args);
  }
}

